/* ==================================================================
   Copyright:   Travellers Tales (UK)
   File:        Material.h
   Author:      Chris Halliday
   Date:        24th Aug 2005
   Format:      Generic C++
   Description: Shared pool based memory manager for physics system.
	
   $Id: Memory.h 35823 2009-02-11 15:24:28Z challiday $
   $Log: not supported by cvs2svn $
   Revision 1.59  2007/10/30 17:33:11  challiday
   Memory leak fix.

===================================================================== */

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
#ifdef DYNOMEMPROFILE
#define IncTag(ClassName,x )  Memory::IncTagCount( #ClassName, sizeof(ClassName),x );
#define DecTag(ClassName,x )		Memory::DecTagCount( #ClassName, sizeof(ClassName),x );
#else
#define IncTag(ClassName,x )
#define DecTag(ClassName,x )
#endif																								

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
#define UseDynoMemory(ClassName)\
	public:\
	void* operator new( size_t size )\
	{\
		IncTag(ClassName,1)\
		return Memory::PoolAlloc<ClassName>();\
	};\
	void  operator delete( void* mem )\
	{\
		DecTag(ClassName,1)\
		Memory::PoolFree<ClassName>( (ClassName*) mem );\
	};\
	void* operator new[]( size_t size )	throw()\
	{\
		void* mem = Memory::SystemAlloc( size );\
		IncTag(ClassName,  NuBlockSize( mem ) / sizeof(ClassName) )\
		return mem;\
	};\
	void  operator delete[](void* mem )\
	{\
		DecTag(ClassName, NuBlockSize( mem )/sizeof(ClassName) )\
		Memory::SystemFree( mem );\
	};
	

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
void Memory::DoAllocTrap( void* mem, int Size )
{
	#ifdef DYNOALLOCTRAP
	int Min = (int) mem;
	int Max = Min + Size;
	if ( Min <= DYNOALLOCTRAP && Max > DYNOALLOCTRAP )
		NuWarning("Alloc Trap");
	#endif
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
void Memory::Lock()
{
	NuThreadCriticalSectionBegin(mCriticalSectionId, 1);		
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
void Memory::UnLock()
{	
	NuThreadCriticalSectionEnd(mCriticalSectionId);		
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////	
void* Memory::SystemAlloc( int Size )
{
	//Allocate the block
	void* Block = AllocFn( Size );

	#if defined	DYNOMEMPROFILE
	UsedSystemMem += NuBlockSize( Block );
	#endif
	
	return Block;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////	
void Memory::SystemFree( void* Mem )
{
	#if defined	DYNOMEMPROFILE
	if ( Mem)
		UsedSystemMem -= NuBlockSize( Mem );	
	#endif
	FreeFn( Mem );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////	
void*	Memory::FastPoolAlloc( int PoolIndex )
{
	//NuAssert( PoolIndex < NumPools, "Pool Index Overflow");

	Lock();	

	//Return some memory from the pool
	void* Return;	
	if ( PoolIndex >= NumPools )
	{
		Return = Memory::SystemAlloc( ChunkSize * (PoolIndex+1) );
	}
	else if ( Pools[ PoolIndex ] )
	{		
		#ifdef DYNOMEMPROFILE
		MemInPools -= (PoolIndex+1) * ChunkSize;
		#endif

		Return = Pools[ PoolIndex ];
		Pools[PoolIndex] = *(void**) Pools[ PoolIndex ];		
	}
	//If the pool is empty get it from the heap.
	else 
	{
		//Return = (void*) Memory::Alloc( ChunkSize * (PoolIndex+1) );	
		int Size = ChunkSize * (PoolIndex+1);
		Return = BufferAlloc( Size );
	}

	DoAllocTrap( Return, ChunkSize * (PoolIndex+1) );
	
	UnLock();
	return Return;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
void Memory::FastPoolFree( void* Mem, int PoolIndex )
{
	//NuAssert( PoolIndex < NumPools, "Pool Index Overflow");

	DoAllocTrap( Mem, ChunkSize * (PoolIndex+1) );

	Lock();

	if (PoolIndex < NumPools )
	{		
		#ifdef DYNOMEMPROFILE
		MemInPools += (PoolIndex+1) * ChunkSize;
		#endif

		*(void**) Mem = Pools[PoolIndex];
		Pools[PoolIndex] = (void*) Mem;
	}
	else
	{
		SystemFree( Mem );//, ChunkSize * (PoolIndex+1) );
	}

	UnLock();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
template <class Type> Type*  Memory::PoolAlloc()
{
	return (Type*) FastPoolAlloc( (sizeof(Type) -1) / ChunkSize );	
}  

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
template <class Type> void Memory::PoolFree( Type *Object )
{
	if (Object)
		FastPoolFree( (void*) Object, (sizeof(Type) -1) / ChunkSize );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
template <class Type > Type*  Memory::PoolAlloc( int Count )
{
	int NumBytes = Count * sizeof( Type );
	Type* Array = (Type*) FastPoolAlloc( ( NumBytes-1) / ChunkSize );
	
	for (int i=0; i<Count; i++)
		new (&Array[i]) Type();

	return Array;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
template <class Type > void Memory::PoolFree( Type *Object, int Count )
{
	int NumBytes = Count * sizeof( Type );
	FastPoolFree( (void*) Object, ( NumBytes-1) / ChunkSize );	
}